<?php
/**
 * RAMBLER QATAR — Interactive Flipbook Catalog Viewer
 * Uses StPageFlip (page-flip.js) for realistic page-turn animation
 * URL: catalog-viewer.php?id=1
 */
$base    = '/ramblerco';
$cats_json = file_get_contents(__DIR__ . '/data/catalogs.json');
$data      = json_decode($cats_json, true);
$all_cats  = $data['catalogs'];

$cat_id  = isset($_GET['id']) ? (int)$_GET['id'] : 1;
$catalog = null;
foreach ($all_cats as $c) { if ($c['id'] === $cat_id) { $catalog = $c; break; } }
if (!$catalog) { $catalog = $all_cats[0]; }

$products = $catalog['products'];
$prev_cat = null; $next_cat = null;
foreach ($all_cats as $c) {
  if ($c['id'] === $cat_id - 1) $prev_cat = $c;
  if ($c['id'] === $cat_id + 1) $next_cat = $c;
}

// Helper: SVG icons for product categories
function getCategoryIcon($sub) {
  $icons = [
    'Writing'=>'<path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M15.232 5.232l3.536 3.536m-2.036-5.036a2.5 2.5 0 113.536 3.536L6.5 21.036H3v-3.572L16.732 3.732z"/>',
    'Stationery'=>'<path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"/>',
    'Awards'=>'<path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M5 3v4M3 5h4M6 17v4m-2-2h4m5-16l2.286 6.857L21 12l-5.714 2.143L13 21l-2.286-6.857L5 12l5.714-2.143L13 3z"/>',
    'Hampers'=>'<path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M20 7l-8-4-8 4m16 0l-8 4m8-4v10l-8 4m0-10L4 7m8 4v10M4 7v10l8 4"/>',
    'default'=>'<path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M21 13.255A23.931 23.931 0 0112 15c-3.183 0-6.22-.62-9-1.745M16 6V4a2 2 0 00-2-2h-4a2 2 0 00-2 2v2m4 6h.01M5 20h14a2 2 0 002-2V8a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z"/>',
  ];
  return $icons[$sub] ?? $icons['default'];
}

// Color helper
$gf = $catalog['gradient_from'];
$gt = $catalog['gradient_to'];
$ac = $catalog['accent'];
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title><?= htmlspecialchars($catalog['title']) ?> — Rambler Qatar Catalogue</title>
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&family=Plus+Jakarta+Sans:wght@400;600;700;800;900&display=swap" rel="stylesheet">
  <!-- StPageFlip -->
  <script src="https://unpkg.com/page-flip/dist/js/page-flip.browser.js"></script>
  <!-- Custom Viewer CSS -->
  <link rel="stylesheet" href="<?= $base ?>/assets/css/style.css">
  <style>
    *,*::before,*::after{box-sizing:border-box;margin:0;padding:0;}
    html,body{height:100%;overflow:hidden;background:#070409;color:#fff;font-family:'Inter',sans-serif;}

    /* ── Top Bar ── */
    #topbar{
      position:fixed;top:0;left:0;right:0;z-index:100;
      display:flex;align-items:center;justify-content:space-between;
      padding:0 20px;height:56px;
      background:rgba(7,4,9,.92);backdrop-filter:blur(20px);
      border-bottom:1px solid rgba(255,255,255,.07);
    }
    #topbar .brand{display:flex;align-items:center;gap:12px;}
    #topbar img{height:32px;}
    #topbar .sep{width:1px;height:20px;background:rgba(255,255,255,.12);}
    #topbar .cat-title{font-size:13px;font-weight:600;color:rgba(255,255,255,.7);}
    #topbar .top-actions{display:flex;align-items:center;gap:8px;}

    /* ── Flipbook Stage ── */
    #stage{
      position:fixed;top:90px;left:0;right:0;bottom:60px;
      display:flex;align-items:center;justify-content:center;
      padding:20px;
      background:radial-gradient(ellipse at 50% 30%, rgba(224,53,64,.06) 0%, transparent 60%),
                 radial-gradient(ellipse at 80% 70%, rgba(28,46,112,.08) 0%, transparent 50%),
                 #070409;
      transition:top .4s;
    }
    /* When hint is dismissed, stage expands up */
    #stage.hint-gone { top:56px; }
    /* Grid overlay */
    #stage::before{
      content:'';position:absolute;inset:0;pointer-events:none;
      background-image:linear-gradient(rgba(224,53,64,.025) 1px,transparent 1px),linear-gradient(90deg,rgba(224,53,64,.025) 1px,transparent 1px);
      background-size:50px 50px;
    }

    /* ── Bottom Controls ── */
    #controls{
      position:fixed;bottom:0;left:0;right:0;z-index:100;
      display:flex;align-items:center;justify-content:center;gap:10px;
      padding:0 20px;height:60px;
      background:rgba(7,4,9,.88);backdrop-filter:blur(20px);
      border-top:1px solid rgba(255,255,255,.07);
    }
    .ctrl-btn{
      display:inline-flex;align-items:center;justify-content:center;
      width:38px;height:38px;border-radius:10px;border:1px solid rgba(255,255,255,.1);
      background:transparent;color:rgba(255,255,255,.5);cursor:pointer;
      transition:all .2s;font-size:14px;
    }
    .ctrl-btn:hover{border-color:rgba(224,53,64,.5);color:#E03540;background:rgba(224,53,64,.08);}
    .ctrl-btn:disabled{opacity:.3;cursor:not-allowed;}
    #page-indicator{
      font-size:12px;font-weight:500;color:rgba(255,255,255,.4);
      min-width:90px;text-align:center;
    }
    #progress-bar{
      position:fixed;bottom:60px;left:0;height:2px;
      background:linear-gradient(to right,#E03540,#C9A84C,#4268D4);
      transition:width .4s cubic-bezier(.4,0,.2,1);z-index:101;
    }

    /* ══════════════════════════════════════
       BOOK / PAGE STYLES
    ══════════════════════════════════════ */
    #flipbook{width:100%;height:100%;}

    /* All pages share this */
    .page{background:#fff;overflow:hidden;user-select:none;}
    .page-cover{cursor:pointer;}

    /* ── COVER PAGE ── */
    .pg-cover{
      width:100%;height:100%;position:relative;
      background:linear-gradient(145deg,<?= $gf ?>,<?= $gt ?>);
      display:flex;flex-direction:column;justify-content:space-between;
      padding:40px 36px;overflow:hidden;
    }
    .pg-cover .dot-grid{
      position:absolute;inset:0;pointer-events:none;
      background-image:radial-gradient(circle at 1px 1px,rgba(255,255,255,.12) 1px,transparent 0);
      background-size:22px 22px;
    }
    .pg-cover .circle-deco{
      position:absolute;right:-15%;top:-10%;
      width:70%;padding-bottom:70%;border-radius:50%;
      background:rgba(255,255,255,.04);pointer-events:none;
    }
    .pg-cover .circle-deco2{
      position:absolute;left:-20%;bottom:-15%;
      width:60%;padding-bottom:60%;border-radius:50%;
      background:rgba(255,255,255,.03);pointer-events:none;
    }

    /* ── INNER PAGES ── */
    .pg-inner{width:100%;height:100%;display:flex;flex-direction:column;}

    /* Left intro page */
    .pg-intro{
      background:#0D0A10;color:#fff;
      width:100%;height:100%;padding:44px 40px;
      display:flex;flex-direction:column;justify-content:space-between;
      position:relative;overflow:hidden;
    }
    .pg-intro::before{
      content:'';position:absolute;top:0;left:0;right:0;height:3px;
      background:linear-gradient(to right,<?= $gf ?>,<?= $gt ?>);
    }
    .pg-intro .intro-bg{
      position:absolute;right:-20%;top:-20%;
      width:70%;padding-bottom:70%;border-radius:50%;
      background:radial-gradient(circle,<?= $gf ?>22,transparent 70%);
      pointer-events:none;
    }

    /* Right intro: stats page */
    .pg-stats{
      background:#fff;
      width:100%;height:100%;padding:44px 40px;
      display:flex;flex-direction:column;justify-content:space-between;
    }
    .pg-stats .stat-row{
      display:flex;flex-direction:column;gap:4px;
      padding:20px 0;border-bottom:1px solid #e5e7eb;
    }
    .pg-stats .stat-row:last-child{border-bottom:none;}

    /* Product spread — left (visual) */
    .pg-product-left{
      width:100%;height:100%;position:relative;overflow:hidden;
      display:flex;flex-direction:column;justify-content:flex-end;
      padding:36px 32px;
    }
    .pg-product-left .prod-bg{position:absolute;inset:0;}
    .pg-product-left .prod-icon{
      position:absolute;top:50%;left:50%;transform:translate(-50%,-50%);
      opacity:.12;
    }

    /* Product spread — right (details) */
    .pg-product-right{
      background:#fff;
      width:100%;height:100%;padding:44px 40px;
      display:flex;flex-direction:column;justify-content:space-between;
    }
    .detail-row{
      display:flex;justify-content:space-between;align-items:center;
      padding:12px 0;border-bottom:1px solid #f1f3f4;font-size:12px;
    }
    .detail-row:last-child{border-bottom:none;}

    /* Back cover */
    .pg-back-cover{
      width:100%;height:100%;position:relative;
      background:linear-gradient(145deg,<?= $gt ?>,#050308);
      display:flex;flex-direction:column;align-items:center;justify-content:center;
      padding:44px;text-align:center;overflow:hidden;
    }
    .pg-back-cover .dot-grid{
      position:absolute;inset:0;pointer-events:none;
      background-image:radial-gradient(circle at 1px 1px,rgba(255,255,255,.06) 1px,transparent 0);
      background-size:18px 18px;
    }

    /* TOC page */
    .pg-toc{
      background:#fff;
      width:100%;height:100%;padding:44px 40px;
    }
    .toc-row{
      display:flex;justify-content:space-between;align-items:center;
      padding:10px 0;border-bottom:1px solid #f1f3f4;
      font-size:12px;color:#374151;
    }
    .toc-dot{
      flex:1;margin:0 8px;border-bottom:1px dotted #d1d5db;
      height:1px;margin-bottom:-2px;
    }

    /* Action btn in cover */
    .cover-btn{
      display:inline-flex;align-items:center;gap:8px;
      padding:10px 22px;border-radius:100px;
      background:rgba(255,255,255,.15);
      border:1px solid rgba(255,255,255,.25);
      color:#fff;font-size:12px;font-weight:600;
      cursor:pointer;text-decoration:none;
      transition:all .25s;backdrop-filter:blur(8px);
    }
    .cover-btn:hover{background:rgba(255,255,255,.25);}

    /* Thumbnail nav */
    #thumb-tray{
      position:fixed;right:0;top:90px;bottom:60px;
      width:110px;z-index:90;
      background:rgba(7,4,9,.75);backdrop-filter:blur(16px);
      border-left:1px solid rgba(255,255,255,.07);
      overflow-y:auto;padding:12px 8px;
      display:flex;flex-direction:column;gap:8px;
      scrollbar-width:none;
      transform:translateX(100%);
      transition:transform .3s cubic-bezier(.4,0,.2,1);
    }
    #thumb-tray.open{transform:translateX(0);}
    #thumb-tray::-webkit-scrollbar{display:none;}
    .thumb-item{
      cursor:pointer;border-radius:6px;overflow:hidden;
      border:1.5px solid transparent;transition:border-color .2s;flex-shrink:0;
    }
    .thumb-item.active{border-color:#E03540;}
    .thumb-item img,.thumb-item .thumb-pg{
      width:100%;display:block;
      aspect-ratio:0.707;background:#1a1025;
      display:flex;align-items:center;justify-content:center;
      font-size:11px;font-weight:600;color:rgba(255,255,255,.35);
    }

    /* Zoom */
    #stage.zoomed{ overflow:auto; align-items:flex-start; justify-content:flex-start; }

    /* Mobile adjustments */
    @media(max-width:768px){
      #stage{padding:8px;}
      #controls{gap:6px;}
    }
  </style>
</head>
<body>

<!-- ── Top Bar ── -->
<div id="topbar">
  <div class="brand">
    <a href="<?= $base ?>/index.php">
      <img src="<?= $base ?>/logo (2).png" alt="Rambler Qatar"
           style="filter:drop-shadow(0 1px 6px rgba(0,0,0,.6));">
    </a>
    <div class="sep"></div>
    <span class="cat-title"><?= htmlspecialchars($catalog['title']) ?></span>
    <span style="margin-left:6px;padding:2px 10px;background:<?= $ac ?>22;color:<?= $ac ?>;border-radius:100px;font-size:10px;font-weight:700;letter-spacing:.08em;text-transform:uppercase;">
      <?= $catalog['year'] ?>
    </span>
  </div>
  <div class="top-actions">
    <!-- Thumbnails toggle -->
    <button class="ctrl-btn" id="thumb-toggle" title="Page thumbnails">
      <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6h16M4 10h16M4 14h16M4 18h16"/>
      </svg>
    </button>

    <!-- ── Open Real PDF in new tab ── -->
    <a href="<?= $base ?>/<?= $catalog['pdf_url'] ?>" target="_blank" rel="noopener"
       title="Open actual PDF in new tab"
       style="display:inline-flex;align-items:center;gap:6px;padding:0 14px;height:36px;border-radius:100px;
              background:linear-gradient(135deg,rgba(201,168,76,.18),rgba(201,168,76,.08));
              border:1px solid rgba(201,168,76,.35);color:#C9A84C;font-size:11px;font-weight:700;
              letter-spacing:.04em;text-decoration:none;white-space:nowrap;
              transition:background .25s,border-color .25s,box-shadow .25s;"
       onmouseover="this.style.background='rgba(201,168,76,.28)';this.style.boxShadow='0 0 16px rgba(201,168,76,.3)'"
       onmouseout="this.style.background='linear-gradient(135deg,rgba(201,168,76,.18),rgba(201,168,76,.08))';this.style.boxShadow=''">
      <svg style="width:13px;height:13px;" fill="none" stroke="currentColor" viewBox="0 0 24 24">
        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 6H6a2 2 0 00-2 2v10a2 2 0 002 2h10a2 2 0 002-2v-4M14 4h6m0 0v6m0-6L10 14"/>
      </svg>
      <span class="hidden sm:inline">Open PDF</span>
    </a>

    <!-- Download PDF -->
    <a href="<?= $base ?>/<?= $catalog['pdf_url'] ?>" download
       class="ctrl-btn hidden md:inline-flex" title="Download PDF">
      <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 10v6m0 0l-3-3m3 3l3-3M3 17v3a1 1 0 001 1h16a1 1 0 001-1v-3"/>
      </svg>
    </a>
    <!-- Back to catalog -->
    <a href="<?= $base ?>/catalog.php" class="ctrl-btn" title="All Catalogues">
      <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
      </svg>
    </a>
  </div>
</div>

<!-- ── Real PDF Hint Banner ── -->
<div id="pdf-hint"
     style="position:fixed;top:56px;left:0;right:0;z-index:95;
            display:flex;align-items:center;justify-content:center;gap:10px;flex-wrap:wrap;
            padding:7px 16px;
            background:linear-gradient(90deg,rgba(12,9,16,.95),rgba(20,12,8,.95));
            border-bottom:1px solid rgba(201,168,76,.2);
            backdrop-filter:blur(12px);
            transition:opacity .4s,transform .4s;
            font-size:11px;color:rgba(255,255,255,.5);">
  <!-- Gold diamond -->
  <span style="color:#C9A84C;font-size:8px;opacity:.7;">◆</span>
  <span>Interactive preview — flipbook generated from product data.</span>
  <!-- Link to open real PDF -->
  <a href="<?= $base ?>/<?= $catalog['pdf_url'] ?>" target="_blank" rel="noopener"
     style="display:inline-flex;align-items:center;gap:4px;color:#C9A84C;font-weight:700;
            text-decoration:none;border-bottom:1px solid rgba(201,168,76,.35);
            padding-bottom:1px;transition:border-color .2s;"
     onmouseover="this.style.borderColor='#C9A84C'" onmouseout="this.style.borderColor='rgba(201,168,76,.35)'">
    <svg style="width:11px;height:11px;" fill="none" stroke="currentColor" viewBox="0 0 24 24">
      <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 6H6a2 2 0 00-2 2v10a2 2 0 002 2h10a2 2 0 002-2v-4M14 4h6m0 0v6m0-6L10 14"/>
    </svg>
    Open original PDF catalogue ↗
  </a>
  <!-- Dismiss -->
  <button onclick="(function(){var h=document.getElementById('pdf-hint');h.style.opacity='0';h.style.pointerEvents='none';setTimeout(function(){h.style.display='none';document.getElementById('stage').classList.add('hint-gone');document.getElementById('thumb-tray').style.top='56px';},400);})()"
          style="position:absolute;right:12px;background:none;border:none;cursor:pointer;
                 color:rgba(255,255,255,.3);font-size:14px;line-height:1;padding:2px 4px;"
          title="Dismiss">✕</button>
</div>

<!-- ── Progress Bar ── -->
<div id="progress-bar" style="width:0%;"></div>

<!-- ── Flipbook Stage ── -->
<div id="stage">
  <div id="flipbook">

    <!-- ════════════════════════════════════
         FRONT COVER (hard page)
    ════════════════════════════════════ -->
    <div class="page page-cover" data-density="hard">
      <div class="pg-cover">
        <div class="dot-grid"></div>
        <div class="circle-deco"></div>
        <div class="circle-deco2"></div>

        <!-- Top: brand mark -->
        <div style="position:relative;z-index:1;">
          <div style="display:flex;align-items:center;gap:8px;margin-bottom:6px;">
            <div style="width:28px;height:2px;background:<?= $ac ?>;border-radius:2px;"></div>
            <span style="font-size:9px;font-weight:700;color:<?= $ac ?>;letter-spacing:.14em;text-transform:uppercase;">Rambler Qatar</span>
          </div>
          <div style="font-size:8px;color:rgba(255,255,255,.35);letter-spacing:.1em;text-transform:uppercase;">Corporate Gifts &amp; Branded Merchandise</div>
        </div>

        <!-- Center: big catalog number -->
        <div style="position:relative;z-index:1;text-align:center;flex:1;display:flex;flex-direction:column;align-items:center;justify-content:center;gap:20px;padding:24px 0;">
          <div style="font-size:120px;font-weight:900;line-height:1;color:rgba(255,255,255,.07);font-family:'Plus Jakarta Sans',sans-serif;position:absolute;right:-10px;top:50%;transform:translateY(-50%);">
            <?= str_pad($catalog['id'],2,'0',STR_PAD_LEFT) ?>
          </div>
          <div style="position:relative;z-index:2;">
            <div style="font-size:9px;font-weight:700;letter-spacing:.16em;color:rgba(255,255,255,.4);text-transform:uppercase;margin-bottom:16px;">2026 Edition</div>
            <h1 style="font-size:28px;font-weight:900;color:#fff;line-height:1.1;font-family:'Plus Jakarta Sans',sans-serif;margin-bottom:12px;">
              <?= htmlspecialchars($catalog['title']) ?>
            </h1>
            <p style="font-size:11px;color:rgba(255,255,255,.55);line-height:1.6;max-width:240px;margin:0 auto;">
              <?= htmlspecialchars(substr($catalog['description'],0,100)) ?>…
            </p>
          </div>
        </div>

        <!-- Bottom: meta + tags -->
        <div style="position:relative;z-index:1;">
          <div style="display:flex;flex-wrap:wrap;gap:6px;margin-bottom:16px;">
            <?php foreach ($catalog['tags'] as $tag): ?>
            <span style="padding:3px 10px;background:rgba(255,255,255,.1);border:1px solid rgba(255,255,255,.15);border-radius:100px;font-size:9px;font-weight:600;color:rgba(255,255,255,.7);"><?= htmlspecialchars($tag) ?></span>
            <?php endforeach; ?>
          </div>
          <div style="display:flex;align-items:center;justify-content:space-between;padding-top:14px;border-top:1px solid rgba(255,255,255,.1);">
            <span style="font-size:9px;color:rgba(255,255,255,.35);"><?= $catalog['products_count'] ?>+ products · <?= $catalog['pages'] ?> pages</span>
            <span style="font-size:9px;font-weight:700;color:<?= $ac ?>;"><?= $catalog['year'] ?></span>
          </div>
        </div>
      </div>
    </div><!-- /cover -->

    <!-- ════════════════════════════════════
         PAGE 2 — Company Intro (dark)
    ════════════════════════════════════ -->
    <div class="page">
      <div class="pg-intro">
        <div class="intro-bg"></div>
        <div style="position:relative;z-index:1;">
          <div style="display:flex;align-items:center;gap:8px;margin-bottom:24px;">
            <div style="width:24px;height:2px;background:<?= $ac ?>;border-radius:2px;"></div>
            <span style="font-size:9px;font-weight:700;color:<?= $ac ?>;letter-spacing:.12em;text-transform:uppercase;">About This Collection</span>
          </div>
          <h2 style="font-size:22px;font-weight:900;color:#fff;line-height:1.15;font-family:'Plus Jakarta Sans',sans-serif;margin-bottom:16px;">
            Curated for<br>Corporate Excellence
          </h2>
          <p style="font-size:11px;color:rgba(255,255,255,.55);line-height:1.75;margin-bottom:24px;">
            <?= htmlspecialchars($catalog['description']) ?>
          </p>
          <p style="font-size:11px;color:rgba(255,255,255,.4);line-height:1.75;">
            Every product in this catalogue is available with your custom branding — including laser engraving, embroidery, screen printing, and full-colour UV printing. Minimum order quantities are flexible to suit teams of any size.
          </p>
        </div>
        <div style="position:relative;z-index:1;border-top:1px solid rgba(255,255,255,.08);padding-top:20px;">
          <div style="font-size:9px;color:rgba(255,255,255,.3);letter-spacing:.08em;text-transform:uppercase;margin-bottom:8px;">Request a Quote</div>
          <div style="font-size:11px;color:rgba(255,255,255,.5);">sales@ramblerco.com · +974 7099 3232</div>
        </div>
      </div>
    </div>

    <!-- ════════════════════════════════════
         PAGE 3 — Stats + Highlights (white)
    ════════════════════════════════════ -->
    <div class="page">
      <div class="pg-stats">
        <div>
          <div style="display:flex;align-items:center;gap:8px;margin-bottom:24px;">
            <div style="width:24px;height:2px;border-radius:2px;" class="" style="background:<?= $gf ?>;"></div>
            <span style="font-size:9px;font-weight:700;letter-spacing:.12em;text-transform:uppercase;color:#6b7280;">Collection Highlights</span>
          </div>
          <h2 style="font-size:22px;font-weight:900;color:#111;font-family:'Plus Jakarta Sans',sans-serif;line-height:1.15;margin-bottom:28px;">
            The Numbers<br>Speak for Themselves
          </h2>

          <?php
          $stats = [
            [$catalog['products_count'].'+', 'Products in this collection', 'From entry-level to premium tier'],
            [$catalog['pages'], 'Pages of inspiration', 'Curated for corporate gifting'],
            ['24h', 'Quote turnaround time', 'From enquiry to detailed proposal'],
            ['MOQ 5+','Minimum order quantity','Flexible for any team size'],
          ];
          foreach ($stats as $s): ?>
          <div class="stat-row">
            <div style="display:flex;justify-content:space-between;align-items:baseline;gap:12px;">
              <span style="font-size:22px;font-weight:900;font-family:'Plus Jakarta Sans',sans-serif;color:#111;"><?= $s[0] ?></span>
              <div style="text-align:right;">
                <div style="font-size:11px;font-weight:600;color:#374151;"><?= $s[1] ?></div>
                <div style="font-size:10px;color:#9ca3af;margin-top:2px;"><?= $s[2] ?></div>
              </div>
            </div>
          </div>
          <?php endforeach; ?>
        </div>
        <div style="border-top:1px solid #f1f3f4;padding-top:16px;">
          <div style="font-size:9px;color:#9ca3af;margin-bottom:8px;text-transform:uppercase;letter-spacing:.08em;">Branding Methods Available</div>
          <div style="display:flex;flex-wrap:wrap;gap:6px;">
            <?php foreach(['Laser Engraving','Embroidery','Screen Print','UV Printing','Debossing'] as $m): ?>
            <span style="padding:3px 9px;background:#f3f4f6;border-radius:100px;font-size:9px;font-weight:600;color:#374151;"><?= $m ?></span>
            <?php endforeach; ?>
          </div>
        </div>
      </div>
    </div>

    <!-- ════════════════════════════════════
         PAGE 4 — Table of Contents (white)
    ════════════════════════════════════ -->
    <div class="page">
      <div class="pg-toc">
        <div style="margin-bottom:28px;">
          <span style="font-size:9px;font-weight:700;letter-spacing:.12em;text-transform:uppercase;color:#9ca3af;">Contents</span>
          <h2 style="font-size:20px;font-weight:900;color:#111;font-family:'Plus Jakarta Sans',sans-serif;margin-top:8px;">Product Index</h2>
        </div>
        <?php $pg = 5; foreach ($products as $i => $p): ?>
        <div class="toc-row">
          <span style="font-weight:600;color:#111;"><?= htmlspecialchars($p['name']) ?></span>
          <div class="toc-dot"></div>
          <span style="color:#9ca3af;flex-shrink:0;"><?= str_pad($pg, 2, '0', STR_PAD_LEFT) ?></span>
        </div>
        <?php $pg += 2; endforeach; ?>
        <div style="margin-top:auto;padding-top:20px;border-top:1px solid #f1f3f4;font-size:9px;color:#d1d5db;text-align:right;">
          ramblerco.com
        </div>
      </div>
    </div>

    <!-- ════════════════════════════════════
         PRODUCT SPREADS (2 pages each)
    ════════════════════════════════════ -->
    <?php foreach ($products as $idx => $product):
      $hue = $idx % 2 === 0 ? $gf : $gt;
      $icon = getCategoryIcon($product['subcategory']);
    ?>

    <!-- Product Left: Visual -->
    <div class="page">
      <div class="pg-product-left">
        <div class="prod-bg" style="background:linear-gradient(145deg,<?= $gf ?>,<?= $gt ?>);"></div>

        <!-- Dot grid -->
        <div style="position:absolute;inset:0;opacity:.07;background-image:radial-gradient(circle at 1px 1px,rgba(255,255,255,.7) 1px,transparent 0);background-size:18px 18px;pointer-events:none;"></div>

        <!-- Decorative circle -->
        <div style="position:absolute;right:-30%;top:-20%;width:80%;padding-bottom:80%;border-radius:50%;background:rgba(255,255,255,.04);pointer-events:none;"></div>

        <!-- Big product number -->
        <div style="position:absolute;top:20px;right:16px;font-size:80px;font-weight:900;color:rgba(255,255,255,.06);font-family:'Plus Jakarta Sans',sans-serif;line-height:1;">
          <?= str_pad($idx+1,2,'0',STR_PAD_LEFT) ?>
        </div>

        <!-- Icon watermark -->
        <div style="position:absolute;top:50%;left:50%;transform:translate(-50%,-50%);opacity:.08;">
          <svg style="width:120px;height:120px;" fill="none" stroke="white" viewBox="0 0 24 24"><?= $icon ?></svg>
        </div>

        <!-- Category label top -->
        <div style="position:relative;z-index:2;position:absolute;top:24px;left:24px;">
          <span style="font-size:8px;font-weight:700;color:<?= $ac ?>;letter-spacing:.14em;text-transform:uppercase;background:<?= $ac ?>22;padding:3px 10px;border-radius:100px;border:1px solid <?= $ac ?>44;">
            <?= htmlspecialchars($product['subcategory']) ?>
          </span>
        </div>

        <!-- Product title bottom -->
        <div style="position:relative;z-index:2;">
          <div style="font-size:8px;font-weight:700;color:rgba(255,255,255,.4);letter-spacing:.1em;text-transform:uppercase;margin-bottom:10px;">
            SKU: <?= htmlspecialchars($product['sku']) ?>
          </div>
          <h3 style="font-size:22px;font-weight:900;color:#fff;line-height:1.1;font-family:'Plus Jakarta Sans',sans-serif;margin-bottom:8px;">
            <?= htmlspecialchars($product['name']) ?>
          </h3>
          <div style="width:32px;height:2px;background:<?= $ac ?>;border-radius:2px;"></div>
        </div>
      </div>
    </div>

    <!-- Product Right: Details -->
    <div class="page">
      <div class="pg-product-right">
        <div>
          <!-- Header -->
          <div style="border-bottom:2px solid <?= $gf ?>;padding-bottom:18px;margin-bottom:22px;">
            <div style="font-size:9px;font-weight:700;color:#9ca3af;letter-spacing:.1em;text-transform:uppercase;margin-bottom:6px;">
              <?= htmlspecialchars($product['subcategory']) ?>
            </div>
            <h3 style="font-size:18px;font-weight:900;color:#111;font-family:'Plus Jakarta Sans',sans-serif;line-height:1.15;margin-bottom:10px;">
              <?= htmlspecialchars($product['name']) ?>
            </h3>
            <p style="font-size:11px;color:#6b7280;line-height:1.7;">
              <?= htmlspecialchars($product['description']) ?>
            </p>
          </div>

          <!-- Details rows -->
          <div class="detail-row">
            <span style="color:#6b7280;font-size:11px;">SKU</span>
            <span style="font-weight:600;color:#111;font-size:11px;font-family:monospace;"><?= htmlspecialchars($product['sku']) ?></span>
          </div>
          <div class="detail-row">
            <span style="color:#6b7280;font-size:11px;">Price From</span>
            <span style="font-weight:800;font-size:14px;font-family:'Plus Jakarta Sans',sans-serif;" style="color:<?= $gf ?>">
              QAR <?= number_format($product['price_from']) ?>
            </span>
          </div>
          <div class="detail-row">
            <span style="color:#6b7280;font-size:11px;">Min. Order Qty</span>
            <span style="font-weight:700;color:#111;font-size:11px;"><?= $product['moq'] ?> units</span>
          </div>
          <div class="detail-row">
            <span style="color:#6b7280;font-size:11px;">Category</span>
            <span style="font-weight:600;color:#111;font-size:11px;"><?= htmlspecialchars($product['subcategory']) ?></span>
          </div>
          <?php if ($product['featured']): ?>
          <div style="margin-top:16px;padding:10px 14px;background:<?= $gf ?>15;border-left:3px solid <?= $gf ?>;border-radius:0 8px 8px 0;">
            <span style="font-size:10px;font-weight:700;text-transform:uppercase;letter-spacing:.08em;" style="color:<?= $gf ?>">★ Featured Product</span>
          </div>
          <?php endif; ?>
        </div>

        <!-- Footer CTA -->
        <div style="border-top:1px solid #f1f3f4;padding-top:18px;">
          <div style="font-size:9px;color:#9ca3af;margin-bottom:6px;text-transform:uppercase;letter-spacing:.08em;">Request This Product</div>
          <div style="font-size:11px;color:#374151;font-weight:500;">sales@ramblerco.com</div>
          <div style="font-size:11px;color:#9ca3af;margin-top:3px;">+974 7099 3232</div>
        </div>
      </div>
    </div>

    <?php endforeach; ?>

    <!-- ════════════════════════════════════
         BACK COVER (hard)
    ════════════════════════════════════ -->
    <div class="page page-cover" data-density="hard">
      <div class="pg-back-cover">
        <div class="dot-grid"></div>
        <div style="position:relative;z-index:1;display:flex;flex-direction:column;align-items:center;gap:20px;">
          <img src="<?= $base ?>/logo (2).png" alt="Rambler Qatar"
               style="height:48px;filter:drop-shadow(0 2px 10px rgba(0,0,0,.5));margin-bottom:8px;">
          <div>
            <h2 style="font-size:18px;font-weight:900;color:#fff;font-family:'Plus Jakarta Sans',sans-serif;margin-bottom:8px;">
              Ready to Order?
            </h2>
            <p style="font-size:11px;color:rgba(255,255,255,.5);line-height:1.7;max-width:220px;">
              Contact our team for a personalised quote within 24 hours.
            </p>
          </div>
          <div style="display:flex;flex-direction:column;gap:8px;margin-top:4px;width:100%;max-width:220px;">
            <div style="display:flex;align-items:center;gap:10px;justify-content:center;">
              <svg style="width:14px;height:14px;color:<?= $ac ?>;" fill="none" stroke="<?= $ac ?>" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 8l7.89 5.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z"/></svg>
              <span style="font-size:11px;color:rgba(255,255,255,.6);">sales@ramblerco.com</span>
            </div>
            <div style="display:flex;align-items:center;gap:10px;justify-content:center;">
              <svg style="width:14px;height:14px;" fill="none" stroke="<?= $ac ?>" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 5a2 2 0 012-2h3.28a1 1 0 01.948.684l1.498 4.493a1 1 0 01-.502 1.21l-2.257 1.13a11.042 11.042 0 005.516 5.516l1.13-2.257a1 1 0 011.21-.502l4.493 1.498a1 1 0 01.684.949V19a2 2 0 01-2 2h-1C9.716 21 3 14.284 3 6V5z"/></svg>
              <span style="font-size:11px;color:rgba(255,255,255,.6);">+974 7099 3232</span>
            </div>
            <div style="display:flex;align-items:center;gap:10px;justify-content:center;">
              <svg style="width:14px;height:14px;" fill="none" stroke="<?= $ac ?>" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 12a9 9 0 01-9 9m9-9a9 9 0 00-9-9m9 9H3m9 9a9 9 0 01-9-9m9 9c1.657 0 3-4.03 3-9s-1.343-9-3-9m0 18c-1.657 0-3-4.03-3-9s1.343-9 3-9m-9 9a9 9 0 019-9"/></svg>
              <span style="font-size:11px;color:rgba(255,255,255,.6);">ramblerco.com</span>
            </div>
          </div>
          <div style="margin-top:12px;font-size:8px;color:rgba(255,255,255,.2);letter-spacing:.12em;text-transform:uppercase;">
            P.O. Box 8898, Doha, Qatar
          </div>
        </div>
      </div>
    </div>

  </div><!-- /#flipbook -->
</div><!-- /#stage -->

<!-- ── Bottom Controls ── -->
<div id="controls">
  <!-- Prev catalog -->
  <?php if ($prev_cat): ?>
  <a href="?id=<?= $prev_cat['id'] ?>" class="ctrl-btn hidden md:inline-flex" title="Previous Catalogue" style="font-size:10px;gap:4px;width:auto;padding:0 10px;">
    <svg class="w-3.5 h-3.5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 17l-5-5m0 0l5-5m-5 5h12"/></svg>
    Prev
  </a>
  <?php else: ?>
  <div style="width:62px;"></div>
  <?php endif; ?>

  <button id="btn-first" class="ctrl-btn" title="First page">
    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 19l-7-7 7-7M19 19l-7-7 7-7"/></svg>
  </button>
  <button id="btn-prev" class="ctrl-btn" title="Previous page">
    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"/></svg>
  </button>

  <div id="page-indicator">Page 1 of —</div>

  <button id="btn-next" class="ctrl-btn" title="Next page">
    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"/></svg>
  </button>
  <button id="btn-last" class="ctrl-btn" title="Last page">
    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 5l7 7-7 7M5 5l7 7-7 7"/></svg>
  </button>

  <!-- Open Real PDF in new tab -->
  <a href="<?= $base ?>/<?= $catalog['pdf_url'] ?>" target="_blank" rel="noopener"
     title="Open full PDF in new tab"
     style="height:38px;display:inline-flex;align-items:center;gap:5px;padding:0 12px;border-radius:10px;
            background:rgba(201,168,76,.12);border:1px solid rgba(201,168,76,.3);
            color:#C9A84C;font-size:11px;font-weight:700;text-decoration:none;transition:all .2s;white-space:nowrap;"
     onmouseover="this.style.background='rgba(201,168,76,.22)';this.style.boxShadow='0 0 14px rgba(201,168,76,.25)'"
     onmouseout="this.style.background='rgba(201,168,76,.12)';this.style.boxShadow=''">
    <svg style="width:13px;height:13px;" fill="none" stroke="currentColor" viewBox="0 0 24 24">
      <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 6H6a2 2 0 00-2 2v10a2 2 0 002 2h10a2 2 0 002-2v-4M14 4h6m0 0v6m0-6L10 14"/>
    </svg>
    <span class="hidden sm:inline">Full PDF</span>
  </a>

  <!-- Download PDF -->
  <a href="<?= $base ?>/<?= $catalog['pdf_url'] ?>" download
     class="ctrl-btn hidden md:inline-flex" title="Download PDF">
    <svg class="w-3.5 h-3.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
      <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 10v6m0 0l-3-3m3 3l3-3M3 17v3a1 1 0 001 1h16a1 1 0 001-1v-3"/>
    </svg>
  </a>

  <!-- Get quote -->
  <a href="<?= $base ?>/index.php#contact"
     style="padding:0 14px;height:38px;display:inline-flex;align-items:center;gap:6px;border-radius:10px;background:linear-gradient(135deg,#FF4D5A,#E03540);color:#fff;font-size:11px;font-weight:700;text-decoration:none;transition:box-shadow .2s;"
     onmouseover="this.style.boxShadow='0 4px 20px rgba(224,53,64,.5)'" onmouseout="this.style.boxShadow=''">
    Get Quote →
  </a>

  <!-- Next catalog -->
  <?php if ($next_cat): ?>
  <a href="?id=<?= $next_cat['id'] ?>" class="ctrl-btn hidden md:inline-flex" title="Next Catalogue" style="font-size:10px;gap:4px;width:auto;padding:0 10px;">
    Next
    <svg class="w-3.5 h-3.5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 7l5 5m0 0l-5 5m5-5H6"/></svg>
  </a>
  <?php else: ?>
  <div style="width:62px;"></div>
  <?php endif; ?>
</div>

<!-- ── Thumbnail Tray ── -->
<div id="thumb-tray">
  <div style="font-size:9px;color:rgba(255,255,255,.3);text-transform:uppercase;letter-spacing:.08em;padding:0 4px 4px;font-weight:600;">Pages</div>
  <!-- Thumbnails generated by JS -->
</div>

<script>
/* ── StPageFlip Init ─────────────────────────── */
const stage = document.getElementById('stage');
const isMobile = window.innerWidth < 768;

// Calculate available dimensions
function getBookSize() {
  const w = stage.offsetWidth  - 48;
  const h = stage.offsetHeight - 48;
  const targetW = isMobile ? Math.min(w, 380) : Math.min(w/2, 520);
  const targetH = Math.min(h, targetW / 0.707);
  return { w: Math.floor(targetW), h: Math.floor(targetH) };
}

const sz = getBookSize();

const pageFlip = new St.PageFlip(document.getElementById('flipbook'), {
  width:    sz.w,
  height:   sz.h,
  size:     'stretch',
  minWidth: 200,
  maxWidth: 560,
  minHeight:280,
  maxHeight:790,
  maxShadowOpacity: 0.6,
  showCover:    true,
  mobileScrollSupport: false,
  clickEventForward:   true,
  usePortrait: isMobile,
  drawShadow:  true,
  flippingTime:900,
  useMouseEvents: true,
  swipeDistance:  40,
  autoSize: true,
});

pageFlip.loadFromHTML(document.querySelectorAll('.page'));

const totalPages = pageFlip.getPageCount();
const indicator  = document.getElementById('page-indicator');
const progBar    = document.getElementById('progress-bar');

function updateUI(page) {
  const p = Math.max(1, page);
  indicator.textContent = `Page ${p} of ${totalPages}`;
  const pct = ((p-1)/(totalPages-1))*100;
  progBar.style.width = pct + '%';
  document.getElementById('btn-prev').disabled  = p <= 1;
  document.getElementById('btn-first').disabled = p <= 1;
  document.getElementById('btn-next').disabled  = p >= totalPages;
  document.getElementById('btn-last').disabled  = p >= totalPages;
  // Update thumb highlight
  document.querySelectorAll('.thumb-item').forEach((t,i)=>{
    t.classList.toggle('active', i === Math.floor((p-1)/2));
  });
}

pageFlip.on('flip',  e => updateUI(e.data + 1));
pageFlip.on('init',  ()=> updateUI(1));

document.getElementById('btn-prev').addEventListener('click',  ()=> pageFlip.flipPrev('bottom'));
document.getElementById('btn-next').addEventListener('click',  ()=> pageFlip.flipNext('bottom'));
document.getElementById('btn-first').addEventListener('click', ()=> pageFlip.flip(0));
document.getElementById('btn-last').addEventListener('click',  ()=> pageFlip.flip(totalPages-1));

// Keyboard nav
document.addEventListener('keydown', e=>{
  if(e.key==='ArrowRight'||e.key==='ArrowDown') pageFlip.flipNext();
  if(e.key==='ArrowLeft'||e.key==='ArrowUp')    pageFlip.flipPrev();
  if(e.key==='Home') pageFlip.flip(0);
  if(e.key==='End')  pageFlip.flip(totalPages-1);
});

// Generate thumbnail tray
const tray = document.getElementById('thumb-tray');
for(let i=0; i<Math.ceil(totalPages/2); i++){
  const d = document.createElement('div');
  d.className='thumb-item';
  d.dataset.page = i*2;
  d.innerHTML=`<div class="thumb-pg">${String(i*2+1).padStart(2,'0')}</div>`;
  d.addEventListener('click',()=>pageFlip.flip(i*2));
  tray.appendChild(d);
}
updateUI(1);

// Thumbnail tray toggle
document.getElementById('thumb-toggle').addEventListener('click',()=>{
  tray.classList.toggle('open');
  // Adjust stage padding when tray opens
  stage.style.paddingRight = tray.classList.contains('open') ? '118px' : '20px';
});

// Window resize
window.addEventListener('resize',()=>{
  if(window.innerWidth<768 && tray.classList.contains('open')){
    tray.classList.remove('open');
    stage.style.paddingRight='20px';
  }
});
</script>
</body>
</html>
